<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Auth\Notifications\ResetPassword;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;

class User extends Authenticatable
{
    use SoftDeletes, Notifiable;

    protected $fillable = [
        'name',
        'email',
		'mobile',
		'country',
        'password',
        'mypass',
        'created_at',
        'updated_at',
        'deleted_at',
        'remember_token',
        'email_verified_at',
		'groupname',
		'afid',
		'affiliate',
		'promo',
        'usermanager',
        'mt4access',
        'deposit',
        'withdrawal',
        'leads',
        'issubadmin'
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $dates = [
        'updated_at',
        'created_at',
        'deleted_at',
        'email_verified_at'
    ];

    public function getEmailVerifiedAtAttribute($value)
    {
        return $value ? Carbon::createFromFormat('Y-m-d H:i:s', $value)->format(config('panel.date_format') . ' ' . config('panel.time_format')) : null;
    }

    public function setEmailVerifiedAtAttribute($value)
    {
        $this->attributes['email_verified_at'] = $value ? Carbon::createFromFormat(config('panel.date_format') . ' ' . config('panel.time_format'), $value)->format('Y-m-d H:i:s') : null;
    }

    public function setPasswordAttribute($input)
    {
        if ($input) {
            $this->attributes['password'] = app('hash')->needsRehash($input) ? Hash::make($input) : $input;
        }
    }

    public function sendPasswordResetNotification($token)
    {
        $this->notify(new ResetPassword($token));
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class);
    }

    protected static function boot()
    {
        parent::boot();

        static::created(function ($model) {
            $user = Auth::user();
            $userRoles = $user ? $user->roles->pluck('title')->implode(', ') : 'guest';

            $attributes = $model->getAttributes();
            $createdAttributes = [];
            foreach ($attributes as $key => $value) {
                if ($key === 'created_at' || $key === 'updated_at') {
                    continue;
                }
                $createdAttributes[] = "$key: '$value'";
            }

            $description = 'Created ' . class_basename($model);
            if (!empty($createdAttributes)) {
                $description .= ' (' . implode(', ', $createdAttributes) . ')';
            }

            Log::create([
                'user_id'      => $user ? $user->id : null,
                'role'         => $userRoles,
                'action'       => 'create',
                'description'  => $description,
                'ip_address'   => request()->ip(),
            ]);
        });

        static::updated(function ($model) {
            $user = Auth::user();
            $userRoles = $user ? $user->roles->pluck('title')->implode(', ') : 'guest';

            $changes = $model->getChanges();
            $original = $model->getOriginal();

            $changedFields = [];
            foreach ($changes as $attribute => $newValue) {
                if ($attribute === 'updated_at') {
                    continue;
                }
                $oldValue = $original[$attribute] ?? null;
                $changedFields[] = "$attribute: '$oldValue' → '$newValue'";
            }

            $description = 'Updated ' . class_basename($model);
            if (!empty($changedFields)) {
                $description .= ' (' . implode(', ', $changedFields) . ')';
            }

            Log::create([
                'user_id'      => $user ? $user->id : null,
                'role'         => $userRoles,
                'action'       => 'update',
                'description'  => $description,
                'ip_address'   => request()->ip(),
            ]);
        });

        static::deleted(function ($model) {
            $user = Auth::user();
            $userRoles = $user ? $user->roles->pluck('title')->implode(', ') : 'guest';

            $attributes = $model->getAttributes();
            $deletedAttributes = [];
            foreach ($attributes as $key => $value) {
                if ($key === 'created_at' || $key === 'updated_at') {
                    continue; 
                }
                $deletedAttributes[] = "$key: '$value'";
            }

            $description = 'Deleted ' . class_basename($model);
            if (!empty($deletedAttributes)) {
                $description .= ' (' . implode(', ', $deletedAttributes) . ')';
            }

            Log::create([
                'user_id'      => $user ? $user->id : null,
                'role'         => $userRoles,
                'action'       => 'delete',
                'description'  => $description,
                'ip_address'   => request()->ip(),
            ]);
        });
    }
}
